## -*-Tcl-*-
 # ==========================================================================
 #  Vince's Additions - an extension package for Alpha
 # 
 #  FILE: "wwwParsing.tcl"
 #                                    created: 04/30/1997 {11:04:46 am} 
 #                                last update: 12/04/2001 {18:19:09 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #     
 #  Description:
 #  
 #  Procedures to fetch remote and then render local html files.
 #  
 #  See the "wwwVersionHistory.tcl" file for license info, credits, etc.
 # ==========================================================================
 ##

# Make sure that the wwwMode.tcl file has been loaded.
wwwMode.tcl

proc wwwParsing.tcl {} {}

namespace eval WWW {}

# ===========================================================================
# 
#  ----  #
# 
#  Rendering Files  #
# 

proc WWW::renderUrl {{url ""} {fileName ""}} {

    global WWWmodeVars WWW::NoUrls WWW::FileNumber file::separator 
    global WWW::UrlSource

    set url [string trim $url {\"}]
    set url [string trim $url]
    if {![string length $url]} {
	if {[llength [winNames]] && [isSelection]} {set url [getSelect]}
	if {![string length $url]} {set url $WWWmodeVars(homePage)}
	if {[set WWW::NoUrls]} {
	    WWW::renderFile
	    return
	} elseif {[catch {dialog::getUrl "Enter the url to be viewed:" $url} url]} {
	    status::errorMsg "Cancelled."
	}
    }
    if {![regexp {^[^:]+:/[/]+} $url]} {set url http://$url} 
    if {[regexp {^file://} $url]} {
	WWW::massagePath  url
	WWW::renderFile  $url
	return
    } elseif {[regexp {^ftp://} $url]} {
	WWW::ftpLink $url
    } elseif {[set WWW::NoUrls]} {
	# We can't handle this yet.
	WWW::externalLink $url
	return
    } 
    # Make sure that this url has a trailing '/' if it is a server.
    set parent [string range $url 0 [string last "/" $url]]
    if {[regexp {^[^:]+:/[/]+$} $parent]} {append url "/"}
    # Check to make sure that we don't already have this window, or at least
    # the file (so that we don't have to download it again.)
    foreach name [array names WWW::UrlSource] {
	if {[set WWW::UrlSource($name)] == $url} {
	    # Hmmm...  is it a window or a file?
	    if {![catch {bringToFront $name}]} {
		return
	    } elseif {[file isfile $name]} {
		WWW::renderFile $name
		return
	    } 
	} 
    }
    if {![string length $fileName]} {
	ensureset WWW::FileNumber 0
	incr WWW::FileNumber
	set fileName "WWW-[set WWW::FileNumber].html"
    } 
    set newFile [file join [temp::directory WWW-tmp] $fileName]
    set WWW::UrlSource([file nativename $newFile]) $url
    status::msg "Fetching $url"
    url::fetch $url [temp::directory WWW-tmp] $fileName
    WWW::renderFile $newFile
}

proc WWW::renderFile {{f ""} {title ""}} {
    
    if {[string length $title] && ![catch {bringToFront $title}]} {
	# We already have this window open.
	return
    }
    if {$f == ""} {set f [getfile "View which file?"]}
    
    global WWW::BaseUrl WWW::UrlSource WWW::FileSource mode WWWmodeVars

    WWW::massagePath f
    set t [file::readAll $f]
    # Is this an html file?
    set pat    "(<((!DOCTYPE\[\r\n\t \]+)?HTML|BODY)\[^>\]*>)"
    set isHtml [regexp -nocase $pat $t]
    # Try to determine what the title should be.
    if {![string length $title]} {
	if {$isHtml} {
	    regexp -nocase {<TITLE[^>]*>(.*)</TITLE>} $t dummy title
	} elseif {[info exists WWW::UrlSource($f)]} {
	    set url [string trimright [set WWW::UrlSource($f)] "/"]
	    regsub {\?.+$} $url {} url
	    set last  [string last "/" $url]
	    set title [string range $url [expr $last + 1] end]
	} else {
	    set title [file tail $f]
	}
    }
    # Massage the title so it works in the menus.
    regsub -all "\[\r\n\t \]+" $title { } title
    set title [win::MakeTitle [quote::Unhtml [string trim $title]]]
    regsub -all {/}        $title {-} title
    regsub -all {[\{\(\<]} $title {} title
    regsub -all {[\}\)\>]} $title {} title
    set title [string trim $title]
    # Determine what the base is for this file.
    set pat {<BASE[\r\n\t ]+HREF[\r\n\t ]*=[\r\n\t ]*"*([^"]*)"*>}
    if {![regexp -nocase $pat $t dummy base]} {
	if {[info exists WWW::UrlSource($f)]} {
	    # This should always supercede.
	    set base [set WWW::UrlSource($f)]
	} else {
	    set base [file::toUrl $f]
	}
    }
    regsub -all {\\} $base {/} base
    # Make sure that we're not dealing with a title that has already been
    # registered, but from a different source.
    if {[info exists WWW::BaseUrl($title)]} {
	if {$base != [set WWW::BaseUrl($title)]} {
	    # 2 titles, different bases.  We'll create a different title
	    # for this page.
	    while {1} {
		if {![info exists "WWW::BaseUrl($title )"]} {
		    set title "$title "
		    break
		} elseif {[set "WWW::BaseUrl($title )"] == $base} {
		    set title "$title "
		    break
		} else {
		    set title "$title "
		}
	    }
	}
    }
    set WWW::BaseUrl($title) $base
    if {![info exists WWW::FileSource($title)]} {
	set WWW::FileSource($title) $f
    }
    if {[info exists WWW::UrlSource($f)]} {
        set WWW::UrlSource($title) [set WWW::UrlSource($f)]
    } 
    regsub -all {\\} [set WWW::BaseUrl($title)] {/} WWW::BaseUrl($title)
    # If this title is not already a window ...
    if {[catch {bringToFront $title}]} {
	# ... should we kill the current window? ...
	if {!$WWWmodeVars(linksOpenNewWindow)} {
	    if {$mode == "WWW" && [win::Current] != "* WWW History *"} {
		killWindow
	    }
	} 
	# ... then create a new shell window, and add the parsed text.
	status::msg "Rendering"
	set title [new -n $title -m "WWW"]
	setWinInfo -w $title shell 1
	if {!$isHtml} {
	    # Not an html file.
	    insertText -w $title $t
	} else {
	    regexp -nocase {<HTML[^>]*>(.*)</HTML>} $t dummy t
	    regexp -nocase {<BODY[^>]*>(.*)</BODY>} $t dummy t
	    catch {WWW::removeCrap t}
	    catch {WWW::parseHtml $t $title}
	    insertText -w $title "\r\r"
	}
	setWinInfo -w $title read-only 1	
	goto [minPos]
	refresh
	status::msg ""
    }
    WWW::visitedLink [set WWW::BaseUrl($title)]
    WWW::addHistoryItem $f
    if {$WWWmodeVars(autoFloatMarks)} {floatNamedMarks}
}

#  Limitations:  Several tags can not be dealt with.
#  Feel free to add to this list, or find a way to deal with them !!

proc WWW::removeCrap {tt} {
    
    global WWWmodeVars

    upvar $tt t
    # Get rid of all scripts and comments.
    WWW::substituteForTags {<SCRIPT} {/SCRIPT>} t
    WWW::substituteForTags {<!--}    {-->}      t
    WWW::substituteForTags {<STYLE}  {/STYLE>}  t
    
    # Get rid of other stuff that we know we can't handle.
    regsub -nocase -all -- {<IMG[ \t\r\n]+SRC[ \t\r\n]*=[ \t\r\n]*\"[^\"]*bullet.gif\"[^>]*>} $t {} t
#   regsub -nocase -all -- "\[ \t\r\n\]+alt=\"(\[^\"\]*)\"\[^>\]*>" $t ">\\1<TD></A>" t
    regsub -nocase -all {<DIV[^>]*>}      $t {} t
    regsub -nocase -all {</DIV>}          $t {} t
    regsub -nocase -all {<FONT[^>]*>}     $t {} t
    regsub -nocase -all {</FONT>}         $t {} t
    regsub -nocase -all {<SPAN[^>]*>}     $t {} t
    regsub -nocase -all {</SPAN>}         $t {} t
    regsub -nocase -all {<TABLE[^>]*>}    $t {} t
    regsub -nocase -all {</TABLE>}        $t {} t
    regsub -nocase -all {<VAR>}           $t {} t
    regsub -nocase -all {</VAR>}          $t {} t
    # How does the user want to handle forms?
    if {$WWWmodeVars(ignoreForms)} {
	WWW::substituteForTags {<FORM} {/FORM>} t
    } else {
	regsub -nocase -all -- {<FORM[^>]*>}  $t {(form begins)} t
	regsub -nocase -all -- {</FORM[^>]*>} $t {(form begins)} t
	WWW::substituteForTags {<SELECT} {/SELECT>} t {(option list)}
    }
    # How does the user want to handle images?
    if {$WWWmodeVars(ignoreImages)} {
	regsub -nocase -all -- {<img[^>]*>} $t {} t
    } else {
	regsub -nocase -all -- "<img\[^>\]*alt\[\r\n\t =\]+\"(\[^\"\]+)\"\[^>\]*>" $t {[\1]} t
	regsub -nocase -all -- {<img[^>]*>} $t {[image]} t
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 #  "WWW::substituteForTags" --
 #  
 #  Substitute for tags that aren't properly supported.
 #  
 # -------------------------------------------------------------------------
 ##

proc WWW::substituteForTags {openTag closeTag _t {substitute ""}} {
    
    upvar $_t t

    set openTag  [string tolower $openTag]
    set closeTag [string tolower $closeTag]
    set OpenTag  [string toupper $openTag]
    set CloseTag [string toupper $closeTag]
    foreach tag [list $openTag $OpenTag] {
	while {[set index0 [string first $tag $t]] != -1} {
	    set tagEnd [string first $closeTag $t]
	    set TagEnd [string first $CloseTag $t]
	    if {$tagEnd < $index0} {set tagEnd "-1"} 
	    if {$TagEnd < $index0} {set TagEnd "-1"} 
	    if {$tagEnd == "-1" && $TagEnd == "-1"} {
		#alertnote "Unbalanced '$tag' tag"
		break
	    } elseif {$tagEnd != "-1" && $TagEnd != "-1"} {
		set index1 [expr {$tagEnd > $TagEnd} ? $TagEnd : $tagEnd]
	    } elseif {$tagEnd == "-1"} {
		set index1 $TagEnd
	    } else {
		set index1 $tagEnd
	    }
	    set index0 [expr $index0 - 1]
	    set index1 [expr $index1 + [string length $closeTag]]
	    set t0     [string range $t 0 $index0]
	    set t1     [string range $t $index1 end]
	    set t      ${t0}${substitute}${t1}
	}
    } 
}

## 
 # -------------------------------------------------------------------------
 # 
 #  "WWW::parseHtml" --
 #  
 #  This is the real workhorse here.
 #  
 # -------------------------------------------------------------------------
 ##

proc WWW::parseHtml {t title} {
    
    if {[string length $t] < 10} {
        # We have a problem here.
        set t "Sorry, nothing found in this file."
    }
    
    global WWW::Indentation WWW::Indent WWWmodeVars WWW::HtmlToStyle
    global WWW::Links WWW::LinksReverse WWW::BaseUrl WWW::Anchors 
    global WWW::Pre WWW::Ext WWW::Marks
    
    catch {unset WWW::Links($title)}
    catch {unset WWW::LinksReverse($title)}
    catch {unset WWW::Anchors($title)}
    
    set WWW::Indentation 0
    set WWW::Indent      ""
    set WWW::Pre 0
    set WWW::Ext 0

    set length1 [string length $t]
    set length2 [string length $t]
    
    # Don't bother giving the stats for Alpha7 -- too blinky and annoying.
    if {[info tclversion] < 8.0} {set msg 0} else {set msg 1}

    # We can't quite handle rendering in background windows, and even
    # if we are in the foreground, it seems as if we really need to
    # force the issue here.
    bringToFront $title
    insertText -w $title \r

    while {[regexp {^([^<]*(<[<>][^<]*)*)<([^<>][^>]*)> *(.*)$} $t dummy first dmy html t]} {
	if {$msg} {
	    set length3 [expr (100 * ($length2 - $length1)) / $length2]
	    status::msg "Rendering -- $length3 %"
	} 
	WWW::wrapInsertText $title $first
	regsub -all {\(} $html {\\\(} html
	switch -regexp [string toupper $html] {
	    "^APPLET" {
		set html [lindex [join $html] 0]
		WWW::split t </APPLET> pre
		if {![regexp -nocase {code *= *([^.]*)\.class} $html dummy class]} {
		    set class "applet"
		}
		WWW::makeLinkWord $title "Run java $class" "\"${class}.java\"" $title
	    }
            "^A\[\r\n\t \]+(\[A-Z0-9\r\n\t =\"\]+)?HREF\[\r\n\t \]*=.*" {
		set pat ".+(HREF\[\r\n\t \]*=\[\r\n\t \"\]*\[^\r\n\t \]+)"
                regexp -nocase $pat $html allofit html
                set where [expr 1 + [string first "=" $html]]
                set href  [string range $html $where end]
                set href  [string trim $href]
                set href  [string trim [string trim $href {\"}]]
		if {[catch {WWW::findClosingTag "</A>" t name}]} {continue}
		regsub -all {<[^>]*>} $name {} name
		set name [string trim $name]
		set base [set WWW::BaseUrl($title)]
		if {[catch {url::makeAbsolute $base $href} link]} {
		    set link ????://$href
		} 
		WWW::makeLinkWord $name $link $title
            }
            "^A\[\r\n\t \]+NAME\[\r\n\t \]*=.*" {
		set anchor [string range $html [expr 1+ [string first "=" $html]] end]
		set anchor [string trim $anchor " \""]
		lappend WWW::Anchors($title) [list $anchor [getPos -w $title]]
	    }
	    "^A(REA)?(\[\r\n\t \]+(SHAPE|COORDS)\[\r\n\t \]*=\[\r\n\t \"\]*\[^\r\n\t \"\]+\"?)*\[\r\n\t \]+HREF\[\r\n\t \]*=.*" {
		regsub -nocase -all "\[\r\n\t \]+TARGET\[\r\n\t \]*=\[\r\n\t \]*\"\[^\"\]*\"" $html {} html
		set html [string range $html [expr 1 + [string last "=" $html]] end]
		if {[regexp -nocase {^([^<]*)(<TD>)?</A>(.*)$} $t "" name td t]} {
		    set pad ""
		    if {[string length $name] == 0} {
			set name [file rootname [string trim $html "\""]]
			set pad "\r\r"
		    } elseif {$td == "<TD>"} {
			set pad "\r"
		    }
		    WWW::makeLinkWord $title $name $html
		    insertText -w $title $pad
		}
	    }
	    "^BR( .*)?" {
		WWW::ensureBreak $title
		if {![set WWW::Pre]} {set t [string trimleft $t]}
	    }
	    "^(B|I|U|BIG|SMALL|EM|STRONG)\$" {
		if {[regexp -nocase "^(\[^<\]*)</$html>(.*)\$" $t "" name t]} {
		    global WWW::HtmlToStyle
		    WWW::makeColourWord $title $name \
		      [set WWW::HtmlToStyle([string toupper $html])]
		}
	    }
	    "^COMMENT" {
		WWW::split t </COMMENT> pre
	    }
	    "^DT" {
		WWW::ensureBreak $title
		insertText -w $title "[string range [set WWW::Indent] 2 end]"
	    }
	    "^DD" {
		WWW::ensureBreak $title
	    }
	    "^/DD" {
		WWW::ensureBreak $title
	    }
	    "^EMBED\[\r\n\t \]+" {
		if {[regexp -nocase {src *= *"([^"]+)"} $html dummy embed]} {
		    set name "???"
		    regexp {[^/:]+$} $embed name
		    WWW::makeLinkWord "Embedded '$name'." $embed
		}
	    }
	    "^HR" {
		WWW::ensureEmptyLine $title
		insertText -w $title "     ________________________________________________________________"
		WWW::ensureEmptyLine $title
	    }			
	    "^H\[0-9\]" {
		WWW::ensureEmptyLine $title
		# Preliminaries
		set oldt $t
		set html [lindex [join $html] 0]
		set num  [string range $html 1 end]
		set num  [expr {[string range $html 1 end] > 2} ? 3 : $num]
		if {$num == "1"} {
		    replaceText -w $title [getPos -w $title] [getPos -w $title] "\r"
		}
		set color $WWWmodeVars(header${num}Color)
		set style $WWWmodeVars(header${num}Style)
		set indent ""
		while {$num > 1} {append indent "  " ; incr num -1}
		set pos0 [getPos -w $title]
		# Note that headings which have links embedded within them
		# will linked but not stylized.  Anchors will be recorded.
		# Can't deal with both embedded at the same time.
		if {[catch {WWW::findClosingTag "</$html>" t headerString}]} {
		    continue
		}
		if {[regexp -nocase "<A\[\r\n\t \]+HREF\[\r\n\t \]*=\[\r\n\t \"\]*(\[^>\]*)\[\r\n\t \]*>(.*)<" $headerString dummy link name]} {
		    # This is a link, so we'll defer
		    if {![string length $name] > 200} {
			# Make sure that nothing funky is going on.
			setNamedMark ${indent}${name} $pos0 $pos0 $pos0
		    } 
		    set t ${headerString}<P></P><P></P>${t}
		    continue
		} elseif {[regexp -nocase "<A\[\r\n\t \]+NAME\[\r\n\t \]*=\[\r\n\t \"\]*(\[^>\]*)\[\r\n\t \]*>.*</A>" $headerString dummy anchor]} {
		    # Make this an anchor
		    set anchor [string trim $anchor {\"}]
		    lappend WWW::Anchors($title) [list $anchor [getPos -w $title]]
		    set anchor [string range $headerString [expr 1+ [string first "=" $headerString]] end]
		    set html [string trim $html " \""]
		} else {
		    regexp -nocase "^(\[^<\]*)</$html>(.*)\$" $oldt dummy name t
		}
		regsub -all "</?\[^>\]+>" $headerString {} name
		if {[info exists name]} {
		    regsub -all "((&nbsp;)+)|(\[\r\n\t \]+)" $name { } name
		    set name [string trim $name]
		    if {[string length $name] > 200} {
		        # Somebody is doing something funky here ...
		        set t ${headerString}${t}
			continue
		    } 
		    if {[string length $name]} {
			WWW::makeColourWord $title $name $color $style
			set name [win::MakeTitle [quote::Unhtml $name]]
			while {[lcontains marks $name]} {append name " "}
			lappend marks $name
			setNamedMark ${indent}${name} $pos0 $pos0 $pos0
			lappend markPositions [list ${indent}${name} $pos0 $pos0 $pos0]
			unset name
		    }
		} 
		WWW::ensureEmptyLine $title
	    }
	    "^LI" {
		WWW::ensureBreak $title
		global WWW::OLcount[set WWW::Indentation]
		if {[info exists WWW::OLcount[set WWW::Indentation]]} {
		    insertText -w $title "[string range [set WWW::Indent] 2 end][set WWW::OLcount[set WWW::Indentation]] "
		    incr WWW::OLcount[set WWW::Indentation]
		} else {
		    insertText -w $title "[string range [set WWW::Indent] 2 end] "
		}
	    }
	    "^PRE" {
		WWW::ensureEmptyLine $title
		set WWW::Pre 1
	    }
	    "^/PRE" {
		set WWW::Pre 0
		WWW::ensureEmptyLine $title
	    }
	    "^P" {
		WWW::ensureEmptyLine $title
		if {![set WWW::Pre]} {set t [string trimleft $t]}
	    }
	    "^TR" {
		WWW::ensureEmptyLine $title
	    }
	    "^/TD" {
		insertText -w $title " "
	    }
	    "^(UL|DL|OL|BLOCKQUOTE)" {
		WWW::ensureEmptyLine $title
		incr WWW::Indentation 3
		append WWW::Indent "   "
		if {[string toupper $html] == "OL"} {
		    global WWW::OLcount[set WWW::Indentation]
		    set WWW::OLcount[set WWW::Indentation] 1
		} elseif {[string toupper $html] == "DL"} {
		    global WWW::DLcount[set WWW::Indentation]
		    set WWW::DLcount[set WWW::Indentation] 1
		}
	    }
	    "^/(UL|DL|OL|BLOCKQUOTE)" {
		WWW::ensureEmptyLine $title
		if {[string toupper $html] == "/OL"} {
		    global WWW::OLcount[set WWW::Indentation]
		    catch {unset WWW::OLcount[set WWW::Indentation]}
		} elseif {[string toupper $html] == "/DL"} {
		    global WWW::DLcount[set WWW::Indentation]
		    catch {unset WWW::DLcount[set WWW::Indentation]}
		}
		incr WWW::Indentation -3
		set WWW::Indent [string range [set WWW::Indent] 3 end]
	    }
	    "^/.*" {
	    }
	    default {
		if {[regexp -nocase "^(\[^<\]*)</$html>(.*)\$" $t dummy name t]} {
		    WWW::wrapInsertText $title $name
		}
	    }
	}
	set length1 [string length $t]
    }
    WWW::wrapInsertText $title $t
    if {[info exists markPositions]} {set WWW::Marks($title) $markPositions}
}

## 
 # -------------------------------------------------------------------------
 # 
 # "WWW::findClosingTag" --
 # 
 # Find the closest closing tag.  We assume that 't' no longer includes the
 # opening tag, although that might not be an issue.
 # 
 # -------------------------------------------------------------------------
 ##

proc WWW::findClosingTag {tag _t _text} {

    upvar $_t    t
    upvar $_text text

    set tagEnd [string first [string tolower "$tag"] $t]
    set TagEnd [string first [string toupper "$tag"] $t]
    if {$tagEnd == "-1" && $TagEnd == "-1"} {
# 	alertnote "Unbalanced '$tag' tag"
	error "Unbalanced '$tag' tag"
    } elseif {$tagEnd != "-1" && $TagEnd != "-1"} {
        set end [expr {$tagEnd > $TagEnd} ? $TagEnd : $tagEnd]
    } elseif {$tagEnd == "-1"} {
        set end $TagEnd
    } else {
        set end $tagEnd
    }
    set text [string range $t 0 [expr {$end - 1}]]
    set t [string range $t [expr {$end + [string length $tag]}] end]
}

proc WWW::ensureBreak {title} {
    set pos0 [getPos -w $title]
    set pos1 [lineStart $pos0]
    if {[pos::compare $pos0 != $pos1]} {
	set line [getText -w $title $pos1 $pos0]
	if {![regexp "^\[\r\n\t \]*$" $line]} {insertText -w $title \r} 
    }
}

proc WWW::ensureEmptyLine {title} {
    
    global WWW::Pre 

    WWW::ensureBreak $title
    set pos0 [lineStart [getPos -w $title]]
    set pos1 [lineStart [pos::math $pos0 - 1]]
    if {![set WWW::Pre] && [pos::compare $pos0 != $pos1]} {
	set line [getText -w $title $pos1 $pos0]
	if {![regexp "^\[\r\n\t \]*$" $line]} {insertText -w $title \r}
    }
}

proc WWW::split {text at prefix} {
    upvar $prefix a
    upvar $text t
    if {[set p [string first $at [string toupper $t]]] == -1} {
	set a $t
	set t ""
    } else {
	set a [string range $t 0 [expr $p -1]]
	set t [string range $t [expr $p + [string length $at]] end]
    }
}

proc WWW::wrapInsertText {title text {ind ""} {with ""}} {
    
    global WWW::Pre WWW::HtmlToAccents WWW::Ext fillColumn
    
    set pos  [getPos -w $title]
    set pos0 $pos
    if {![set WWW::Pre]} {set text [string trimleft $text]} 

    # Replace some common style elements first.
    regsub -all {&\#160;|&nbsp;} $text { }  text
    regsub -all {&amp;}          $text {\&} text
    regsub -all {&lt;}           $text "<"  text
    regsub -all {&gt;}           $text ">"  text
    regsub -all {&quot;}         $text {"}  text
    regsub -all {&\#8226;}       $text {}  text
    # Replace all other style elements
    set loop 0
    while {[regexp {&([^\r\n\t ;]+);} $text dummy accent]} {
	if {[info exists WWW::HtmlToAccents($accent)]} {
	    regsub -all $dummy $text [set WWW::HtmlToAccents($accent)] text
	} else {
	    if {[regsub -all {&} $accent {\\&} accent]} {
		beep
		status::msg "There is a syntax error in &$accent;."
	    } else {
		set accent "\\&;$accent;"
		regsub -all $dummy $text $accent text
	    }
	    incr loop
	} 
	if {$loop > 100} {
	    break
	    status::msg "There are too many unknown characters or a problem."
	}
    }
    if {$loop > 0} {regsub -all {&;} $text {\&} text}
    # Perform some initial tests.
    if {$text == ""} {
	# Nothing here!
	return [list $pos0 $pos0]
    } elseif {[set WWW::Pre]} {
	# This is preformatted, so simply insert.
	insertText -w $title $text
	return [list $pos0 [getPos -w $title]]
    } else {
	regsub -all "\[\t\r\n \]+" $text " " text
    }
    # Prepare the text for wrapping.
    set ext 0
    set char [lookAt [pos::math [getPos -w $title] - 1]]
    if {[regexp {\r|\n} $char]} {set WWW::Ext 0}
    if {[regexp {bold|outline|shadow} $ind] || [regexp {bold|outline|shadow} $ind]} {
	set ext 1
    } 
    set x [lindex [posToRowCol [getPos -w $title]] 1]
    global WWW::Indentation WWW::Indent
    set fc [expr {$fillColumn - [expr {(3 + [set WWW::Ext]) / 6}]}]
    if {$x > $fc} {
	insertText -w $title "\r[set WWW::Indent]"
	set x 0
	set WWW::Ext 0
    }
    if {$x == 0} {
	insertText -w $title "[set WWW::Indent]"
	incr x [set WWW::Indentation]
    } elseif {[regexp {^(\w|\(|\-)} $text]} {
	if {[regexp {\w} [lookAt [pos::math [getPos -w $title] - 1]]]} {
	    insertText -w $title " "
	    incr x
	}
    }
    set fc [expr {$fc - $x + 1}]
    if {$ext} {set fc [expr {(5 + 6 * $fc) / 7}]}
    set lword [string first " " $text]
    if {$lword == -1} {set lword [string length $text]}
    if {$fc < $lword} {
	set tt ""
	set a [lookAt [pos::math [getPos -w $title] - 2]]
	set b [lookAt [pos::math [getPos -w $title] - 1]]
	if {[regexp {[^\w]} $b] && $a == " "} {
	    set tt $b
# 	    replaceText $b [getPos -w $title] ""
	    backSpace
	}
	insertText -w $title "\r[set WWW::Indent]$tt"
	set WWW::Ext 0
	set pos [getPos -w $title]
	set fc [expr $fillColumn - [set WWW::Indentation]]
	if {$ext} {
	    set fc [expr {(5 + 6 * $fc) / 7}]
	    set WWW::Ext [string length $text]
	}
    }
    # Now we insert the text, 
    while {[string length $text] > $fc} {
	set f [string last " " [string range $text 0 $fc]]
	if {$f == -1} {set f $fc}
	insertText -w $title "[string range $text 0 $f]\r[set WWW::Indent]"
	set WWW::Ext 0
	set text [string range $text [incr f] end]
	set fc [expr $fillColumn - [set WWW::Indentation]]
	if {$ext} {set fc [expr {(5 + 6 * $fc) / 7}]}
    }
    if {$ext} {set WWW::Ext [expr {[set WWW::Ext] + [string length $text]}]}
    insertText -w $title $text
    return [list $pos0 [getPos -w $title]]
}

proc WWW::makeColourWord {title word ind {with ""}} {

    set pos0 [getPos -w $title]
    WWW::wrapInsertText $title $word $ind $with
    # Attempt to avoid underlining, colorizing across empty space.
    set pos1 [lineStart $pos0]
    set pos2 [nextLineStart $pos1]
    set pos3 [pos::math $pos2 - 1]
    if {[regexp {^[\t ]*[\r\n]+} [getText -w $title $pos0 $pos2]]} {
	set line [getText -w $title $pos2 [getPos -w $title]]
	set len  [string length [string trim $line]]
	set pos0 [pos::math $pos2 + [string length $line] - $len]
    } elseif {[pos::compare $pos0 == $pos1]} {
	set line [getText -w $title $pos0 $pos3]
	set len  [string length [string trim $line]]
	set pos0 [pos::math $pos0 + [string length $line] - $len]
    }
    set pos4 [getPos -w $title]
    text::color $pos0 $pos4 $ind
    if {[string length $with]} {
	text::color $pos0 $pos4 $with
    }
}

proc WWW::makeLinkWord {word link {title ""}} {

    global WWW::Links WWW::HtmlToAccents

    while {[regexp {&([^\r\n\t ;]+);} $link dummy accent]} {
	if {[info exists WWW::HtmlToAccents($accent)]} {
	    regsub -all $dummy $link [set WWW::HtmlToAccents($accent)] link
	} else {
	    if {[regsub -all {&} $accent {\\&} accent]} {
		beep
		status::msg "There is a syntax error in &$accent;."
	    } else {
		set accent "\\&;$accent;"
		regsub -all $dummy $link $accent link
	    }
	}
    }

    set word [string trim $word]
    if {![string length $word]} {return}
    
    if {[regexp {\w} [lookAt [pos::math [getPos -w $title] - 1]]]} {
	insertText -w $title " "
    }
    set pos0 [getPos -w $title]
    WWW::wrapInsertText $title [string trim $word]
    # Attempt to avoid hyperlinking, colorizing across empty space.
    set pos1 [lineStart $pos0]
    set pos2 [nextLineStart $pos1]
    set pos3 [pos::math $pos2 - 1]
    if {[regexp {^[\t ]*[\r\n]+} [getText -w $title $pos0 $pos2]]} {
	set line [getText -w $title $pos2 [getPos -w $title]]
	set len  [string length [string trim $line]]
	set pos0 [pos::math $pos2 + [string length $line] - $len]
    } elseif {[pos::compare $pos0 == $pos1]} {
	set line [getText -w $title $pos0 $pos3]
	set len  [string length [string trim $line]]
	set pos0 [pos::math $pos0 + [string length $line] - $len]
    }
    set pos4 [getPos -w $title]
    set cmd  [WWW::makeLink $title $pos0 $pos4 $link]
    if {[string length $title]} {
	lappend WWW::Links($title) [list $pos0 $pos4 $cmd $link]
    } 
    # Insert an extra space if this was an image.
    if {[regexp {^\[.*\]$} $word]} {insertText -w $title " "}
}

proc WWW::makeLink {title pos0 pos1 link} {

    global WWWmodeVars WWW::Visited

    set link [string trim $link]
    set link [string trim $link {\"}]
    if {![lcontains WWW::Visited $link]} {
	set color $WWWmodeVars(linkColor)
    } else {
	set color $WWWmodeVars(visitedLinkColor)
    }
    set cmd "WWW::link \"${link}\""
    text::color $pos0 $pos1 $color
    text::hyper $pos0 $pos1 $cmd
    return $cmd
}

# ===========================================================================
# 
# .